﻿// ◇
// fds: fdx68 selector
// (C)2020 GORRY.

#include "FDSSystem.h"


FDSSystem::FDSSystem()
{
	setRootDir("");
	initialize();
}

FDSSystem::FDSSystem(const char* rootDir)
{
	setRootDir(rootDir);
	initialize();
}

FDSSystem::~FDSSystem()
{
}

void
FDSSystem::initialize()
{
	mCurDir = "";
	mFiles.clear();
}

void
FDSSystem::setRootDir(const std::string& rootDir)
{
	setRootDir(rootDir.c_str());
}

void
FDSSystem::setRootDir(const char* rootDir)
{
	if ((rootDir == NULL) || (rootDir[0] == '\0')) {
		rootDir = "./";
	}
	mRootDir = rootDir;
	if (mRootDir[mRootDir.length()-1] != '/') {
		mRootDir += "/";
	}
	mCurDir = "";
}

void
FDSSystem::setCurDir(const std::string& curDir)
{
	setCurDir(curDir.c_str());
}

void
FDSSystem::setCurDir(const char* curDir)
{
	mCurDir = curDir;
	if (!mCurDir.empty()) {
		if (mCurDir[mCurDir.length()-1] != '/') {
			mCurDir += "/";
		}
	}
}

void
FDSSystem::start()
{
	char buf[256];

	initView();
	setViewLayout();

	{
		attron(COLOR_PAIR(ColorPair::Header));
		memset(buf, ' ', _countof(buf));
		const char* s1 = "FDS: FDX68 File Selector";
		int l1 = strlen(s1);
		const char* s2 = FDS_VERSION " GORRY";
		int l2 = strlen(s2);
		memcpy(buf+COLS/2-l1/2, s1, l1);
		memcpy(buf+COLS-1-l2, s2, l2);
		buf[COLS] = '\0';
		mvaddstr(0, 0, buf);
		attroff(COLOR_PAIR(ColorPair::Header));
		refresh();
	}

	loadIniFile();

	DlgSelect::ItemsVec items;
	int n = mIniFile.getInt("GLOBAL", "SYSTEMS");
	for (int i=0; i<n; i++) {
		sprintf(buf, "SYSTEM-%d", i+1);
		items.push_back(mIniFile.getString(buf, "NAME"));
	}
	items.push_back("[ Quit ]");
	int sel = 0;
	while (!0) {
		{
			DlgSelect dlg;
			dlg.setItemsVec(items);
			dlg.setHeader("[Select System]");
			sel = dlg.start(sel);
			if (sel < 0) {
				continue;
			}
			if (sel == (int)items.size()-1) {
				break;
			}
			sprintf(buf, "SYSTEM-%d", sel+1);
			mIniSecSystem = buf;
		}

		{
			std::string cmd = mIniFile.getString("GLOBAL", "FDDEMUCMD");
			std::string option = mIniFile.getString(mIniSecSystem, "FDDEMUOPT");
			mFddEmu.setCmd(cmd);
			mFddEmu.setOption(option);
			mFddEmu.run();

			setRootDir(mIniFile.getString(mIniSecSystem, "ROOTDIR"));
			mainLoop();

			mFddEmu.kill();
		}
	}

	destroyView();
}

void
FDSSystem::mainLoop()
{
	mFiles.setPath(mRootDir+mCurDir);
	mFiles.getFiles(mCurDir.empty());
	mFiles.sortFiles();

	pathViewCreateWindow();
	pathViewRefresh();
	infoViewCreateWindow();
	infoViewRefresh();
	fddViewCreateWindow();
	fddViewRefresh();
	helpViewCreateWindow();
	helpViewRefresh();

	filerViewCreateWindow();
	filerViewSetPos(0);

	nodelay(mwFilerView, true);
	wtimeout(mwFilerView, 500);

	bool finish = false;
	bool filerRefresh = true;
	while (!finish) {
		int key;
		if (filerRefresh) {
			filerViewRefresh();
		}
		filerRefresh = true;
		key = wgetch(mwFilerView);
		if (key < 256) {
			key = toupper(key);
		}
		switch (key) {
		  case ERR:
			fddViewRefresh();
			filerRefresh = false;
			break;
		  case 0x1b: // ESC
			wtimeout(mwFilerView, 0);
			if (doEscKey(mwFilerView)) {
				finish = true;
			}
			wtimeout(mwFilerView, 500);
			break;
		  case KEY_UP:
			filerViewUpCursor();
			break;
		  case KEY_DOWN:
			filerViewDownCursor();
			break;
		  case 10: // ENTER
			filerViewSelectEntry();
			break;
		  case 8: // BS
		  case KEY_BACKSPACE:
			filerViewBackDir();
			break;
		  case KEY_PPAGE:
			filerViewPageUpCursor();
			break;
		  case KEY_NPAGE:
			filerViewPageDownCursor();
			break;
		  case KEY_HOME:
			filerViewPageTopCursor();
			break;
		  case KEY_END:
			filerViewPageBottomCursor();
			break;
		  case '\\':
		  case '/':
			filerViewRootDir();
			break;

		  case '0':
			filerViewCmdEjectAllDrive();
			break;
		  case '1':
			break;
		  case '2':
			break;

		  case 'A':
		  case ' ':
			filerViewCmdAutoSet();
			break;
		  case 'I':
			filerViewShowFileInfo();
			filerRefresh = false;
			break;
		  case 'R':
			filerViewCmdRename();
			break;

		  default:
			// fprintf(stderr, "key=%d\n", key);
			break;
		}
	}
	filerViewDestroyWindow();
	pathViewDestroyWindow();
}

void
FDSSystem::end()
{
}

void
FDSSystem::initView()
{
	initscr();
	start_color();
	cbreak();
	noecho();
	curs_set(0);

#if 0
	use_default_colors();

	#define MyColor(r,g,b) (((r)*36)+((g)*6)+(b)+16)

	init_pair((short)ColorPair::Normal,             MyColor(4,4,4), MyColor(0,2,1));
	init_pair((short)ColorPair::FilerParentDir,     MyColor(0,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::FilerDir,           MyColor(0,5,0), MyColor(0,2,1));
	init_pair((short)ColorPair::FilerFdxFile,       MyColor(5,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::FilerOtherFile,     MyColor(3,3,3), MyColor(0,2,1));
	init_pair((short)ColorPair::FilerParentDirCsr,  MyColor(0,5,5), MyColor(0,0,5));
	init_pair((short)ColorPair::FilerDirCsr,        MyColor(0,5,0), MyColor(0,0,5));
	init_pair((short)ColorPair::FilerFdxFileCsr,    MyColor(5,5,5), MyColor(0,0,5));
	init_pair((short)ColorPair::FilerOtherFileCsr,  MyColor(3,3,3), MyColor(0,0,5));
	init_pair((short)ColorPair::Header,             MyColor(0,0,0), MyColor(5,3,0));
	init_pair((short)ColorPair::PathHeader,         MyColor(0,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::PathRoot,           MyColor(3,5,0), MyColor(0,2,1));
	init_pair((short)ColorPair::PathCurrent,        MyColor(0,5,3), MyColor(0,2,1));
	init_pair((short)ColorPair::SelectHeader,       MyColor(0,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::SelectItem,         MyColor(5,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::SelectCursor,       MyColor(5,5,5), MyColor(0,0,5));
	init_pair((short)ColorPair::InputHeader,        MyColor(0,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::InputEdit,          MyColor(5,5,5), MyColor(0,2,1));
	init_pair((short)ColorPair::FddHeaderOff,       MyColor(0,0,0), MyColor(2,2,2));
	init_pair((short)ColorPair::FddHeaderOn,        MyColor(5,5,5), MyColor(2,2,2));
	init_pair((short)ColorPair::FddProtect,         MyColor(5,5,5), MyColor(0,0,0));
	init_pair((short)ColorPair::FddCluster,         MyColor(0,5,3), MyColor(0,0,0));
	init_pair((short)ColorPair::HelpHeader,         MyColor(0,5,5), MyColor(0,2,1));
#endif

#if 1
	init_pair((short)ColorPair::Normal,             COLOR_WHITE, COLOR_BLACK);
	init_pair((short)ColorPair::FilerParentDir,     COLOR_YELLOW, COLOR_BLACK);
	init_pair((short)ColorPair::FilerDir,           COLOR_GREEN, COLOR_BLACK);
	init_pair((short)ColorPair::FilerFdxFile,       COLOR_WHITE, COLOR_BLACK);
	init_pair((short)ColorPair::FilerOtherFile,     COLOR_WHITE, COLOR_BLACK);
	init_pair((short)ColorPair::FilerParentDirCsr,  COLOR_YELLOW, COLOR_BLUE);
	init_pair((short)ColorPair::FilerDirCsr,        COLOR_GREEN, COLOR_BLUE);
	init_pair((short)ColorPair::FilerFdxFileCsr,    COLOR_WHITE, COLOR_BLUE);
	init_pair((short)ColorPair::FilerOtherFileCsr,  COLOR_WHITE, COLOR_BLUE);
	init_pair((short)ColorPair::Header,             COLOR_BLACK, COLOR_YELLOW);
	init_pair((short)ColorPair::PathHeader,         COLOR_CYAN, COLOR_BLACK);
	init_pair((short)ColorPair::PathRoot,           COLOR_WHITE, COLOR_BLACK);
	init_pair((short)ColorPair::PathCurrent,        COLOR_GREEN, COLOR_BLACK);
	init_pair((short)ColorPair::SelectHeader,       COLOR_CYAN, COLOR_BLACK);
	init_pair((short)ColorPair::SelectItem,         COLOR_WHITE, COLOR_BLACK);
	init_pair((short)ColorPair::SelectItemCursor,   COLOR_WHITE, COLOR_BLUE);
	init_pair((short)ColorPair::InputHeader,        COLOR_CYAN, COLOR_BLACK);
	init_pair((short)ColorPair::InputEdit,          COLOR_WHITE, COLOR_BLACK);
	init_pair((short)ColorPair::FddHeaderOff,       COLOR_BLACK, COLOR_WHITE);
	init_pair((short)ColorPair::FddHeaderOn,        COLOR_GREEN, COLOR_WHITE);
	init_pair((short)ColorPair::FddProtect,         COLOR_RED, COLOR_BLACK);
	init_pair((short)ColorPair::FddCluster,         COLOR_GREEN, COLOR_BLACK);
	init_pair((short)ColorPair::HelpHeader,         COLOR_CYAN, COLOR_BLACK);
#endif
}

void
FDSSystem::destroyView()
{
	endwin();
}


void
FDSSystem::loadIniFile()
{
	mIniFile.load("fds.ini");
}

void
FDSSystem::setViewLayout()
{
	// 画面全体
	const XYWH fullViewXYWH = {
	  0,
	  1,
	  COLS,
	  LINES-1,
	};

	// 画面全体の右端19桁
	mHelpViewXYWH.W = 19;
	mHelpViewXYWH.H = fullViewXYWH.h();
	mHelpViewXYWH.X = fullViewXYWH.r()-mHelpViewXYWH.w();
	mHelpViewXYWH.Y = fullViewXYWH.y();

	// 残りの上辺4行
	mFddViewXYWH.W = fullViewXYWH.w() - mHelpViewXYWH.w();
	mFddViewXYWH.H = 4;
	mFddViewXYWH.X = fullViewXYWH.x();
	mFddViewXYWH.Y = fullViewXYWH.y();

	// 残りの上辺4行（上辺重なり）
	mPathViewXYWH.W = mFddViewXYWH.w();
	mPathViewXYWH.H = 4;
	mPathViewXYWH.X = fullViewXYWH.x();
	mPathViewXYWH.Y = fullViewXYWH.y() + mFddViewXYWH.h() - 1;

	// 残りの下辺6行（上辺重なり）
	mInfoViewXYWH.W = mFddViewXYWH.w();
	mInfoViewXYWH.H = 6;
	mInfoViewXYWH.X = fullViewXYWH.x();
	mInfoViewXYWH.Y = fullViewXYWH.b() - mInfoViewXYWH.h();

	// 残り（上辺・下辺重なり）
	mFilerViewXYWH.W = mFddViewXYWH.w();
	mFilerViewXYWH.H = mInfoViewXYWH.y() - mPathViewXYWH.b() + 2;
	mFilerViewXYWH.X = fullViewXYWH.x();
	mFilerViewXYWH.Y = mPathViewXYWH.b() - 1;
}

const std::string&
FDSSystem::getDriveName(int id)
{
	static std::string sDriveName[2];

	if (sDriveName[id].empty()) {
		for (int i=0; i<mFddEmu.Drives; i++) {
			char buf[64];
			sprintf(buf, "DRIVE%dNAME", i);
			sDriveName[i] = mIniFile.getString(mIniSecSystem, buf);
		}
	}

	return sDriveName[id];
}

const FDSSystem::ESCKEYMAP FDSSystem::sEscKeyMap[] = {
	{ "[11~", 265 },
	{ "[12~", 266 },
	{ "[13~", 267 },
	{ "[14~", 268 },
	{ "[1~", KEY_HOME },
	{ "[4~", KEY_END },
	{ "Ol", '+' },
	{ "On", '.' },
	{ "Op", '0' },
	{ "Oq", '1' },
	{ "Or", '2' },
	{ "Os", '3' },
	{ "Ot", '4' },
	{ "Ou", '5' },
	{ "Ov", '6' },
	{ "Ow", '7' },
	{ "Ox", '8' },
	{ "Oy", '9' },
};

bool
FDSSystem::doEscKey(WINDOW* window)
{
	std::string keys;
	int key2 = wgetch(window);
	if (key2 == ERR) {
		return true;
	} else {
		keys.push_back(key2);
		while ((key2 = wgetch(window)) != ERR) {
			keys.push_back(key2);
		}
		for (int i=0; i<(int)_countof(sEscKeyMap); i++) {
			if (keys == sEscKeyMap[i].keys) {
				ungetch(sEscKeyMap[i].newkey);
				return false;
			}
		}
		// fprintf(stderr, "keys=[%s]\n", keys.c_str());
	}
	return false;
}

bool
FDSSystem::doEscKeyW(WINDOW* window)
{
	std::wstring wkeys;
	wint_t wch;
	int wkey = wget_wch(window, &wch);
	if (wkey == ERR) {
		return true;
	} else {
		wkeys.push_back((wchar_t)wch);
		while ((wkey = wget_wch(window, &wch)) != ERR) {
			wkeys.push_back((wchar_t)wch);
		}
		std::string keys = WStrUtil::wstr2str(wkeys);
		for (int i=0; i<(int)_countof(sEscKeyMap); i++) {
			if (keys == sEscKeyMap[i].keys) {
				unget_wch((wchar_t)sEscKeyMap[i].newkey);
				return false;
			}
		}
		// fprintf(stderr, "wkeys=[%ls]\n", wkeys.c_str());
	}
	return false;
}


// [EOF]
